import Foundation
import CoreData

/// Core Data stack for LifeClock app
/// Manages persistent store for health data, predictions, and user settings
class CoreDataStack: ObservableObject {
    static let shared = CoreDataStack()
    
    /// The persistent container for the Core Data stack
    lazy var persistentContainer: NSPersistentContainer = {
        let container = NSPersistentContainer(name: "LifeClock")
        
        // Configure for app group sharing between iOS and watchOS
        let storeURL = FileManager.default.containerURL(forSecurityApplicationGroupIdentifier: "group.com.lifeclock.app")?
            .appendingPathComponent("LifeClockModel.sqlite")
        
        if let storeURL = storeURL {
            let description = NSPersistentStoreDescription(url: storeURL)
            description.shouldInferMappingModelAutomatically = true
            description.shouldMigrateStoreAutomatically = true
            container.persistentStoreDescriptions = [description]
        }
        
        container.loadPersistentStores { _, error in
            if let error = error as NSError? {
                fatalError("Core Data error: \(error), \(error.userInfo)")
            }
        }
        
        container.viewContext.automaticallyMergesChangesFromParent = true
        
        return container
    }()
    
    /// The managed object context for the view layer
    var viewContext: NSManagedObjectContext {
        return persistentContainer.viewContext
    }
    
    /// Background context for data operations
    var backgroundContext: NSManagedObjectContext {
        return persistentContainer.newBackgroundContext()
    }
    
    private init() {}
    
    /// Saves the context if there are changes
    func save() {
        let context = persistentContainer.viewContext
        
        if context.hasChanges {
            do {
                try context.save()
            } catch {
                let nsError = error as NSError
                fatalError("Core Data save error: \(nsError), \(nsError.userInfo)")
            }
        }
    }
    
    /// Performs a background save operation
    func saveInBackground(_ block: @escaping (NSManagedObjectContext) -> Void) {
        let context = backgroundContext
        context.perform {
            block(context)
            if context.hasChanges {
                do {
                    try context.save()
                } catch {
                    print("Background save error: \(error)")
                }
            }
        }
    }
}